<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Compra;
use App\Models\ItemPurchase;
use App\Helpers\StockMove;
use App\Services\NFeEntradaService;
use App\Models\ConfigNota;
use App\Models\NaturezaOperacao;
use NFePHP\DA\NFe\Danfe;
use App\Models\ItemCompra;
use App\Models\Produto;
use App\Models\CompraReferencia;
use NFePHP\DA\NFe\Daevento;
use Mail;
use App\Models\EscritorioContabil;
use Dompdf\Dompdf;

class PurchaseController extends Controller
{   
    protected $empresa_id = null;
    public function __construct(){
        $this->middleware(function ($request, $next) {
            $this->empresa_id = $request->empresa_id;
            $value = session('user_logged');
            if(!$value){
                return redirect("/login");
            }
            return $next($request);
        });
    }

    public function index(){
        $totalRegistros = count(Compra::where('empresa_id', $this->empresa_id)->get());
        $compras = Compra::
        orderBy('id', 'desc')
        ->where('empresa_id', $this->empresa_id)
        ->paginate(15);

        $somaCompraMensal = $this->somaCompraMensal();
        return view('compraManual/listAll')
        ->with('compras', $compras)
        ->with('somaCompraMensal', $somaCompraMensal)
        ->with('links', true)
        ->with('graficoJs', true)
        ->with('title', 'Compras');
        
    }

    public function pesquisa(Request $request){
        $compras = Compra::pesquisaProduto($request->pesquisa);
        $totalRegistros = count($compras);

        $somaCompraMensal = $this->somaCompraMensal();
        return view('compraManual/listAll')
        ->with('compras', $compras)
        ->with('somaCompraMensal', $somaCompraMensal)
        ->with('graficoJs', true)
        ->with('title', 'Pequisa de Produto em Compras');
        
    }

    private function somaCompraMensal(){
        $compras = Compra::
        where('empresa_id', $this->empresa_id)
        ->get();
        $temp = [];
        $soma = 0;
        $mesAnterior = null;
        $anoAnterior = null;

        foreach($compras as $key => $c){
            $date = $c->created_at;
            $mes = substr($date, 5, 2);
            $ano = substr($date, 0, 4);


            if($mesAnterior != $mes){
                $temp["Mes: ".$mes."/$ano"] = $c->valor;
            }else{
                $temp["Mes: ".$mesAnterior."/$anoAnterior"] += $c->valor;

            }
            $mesAnterior = $mes;
            $anoAnterior = $ano;
        }

        return $temp;
    }

    

    private function somaCompraMensalFiltro($compras){
        $temp = [];
        $soma = 0;
        $mesAnterior = null;
        $anoAnterior = null;


        foreach($compras as $c){
            $date = $c->created_at;
            $mes = substr($date, 5, 2);
            $ano = substr($date, 0, 4);

            if($mesAnterior != $mes){
                $temp["Mes: ".$mes."/$ano"] = $c->valor;
            }else{
                $temp["Mes: ".$mesAnterior."/$anoAnterior"] += $c->valor;
            }
            $mesAnterior = $mes;
            $anoAnterior = $ano;
        }

        return $temp;
    }

    private function somaCompraDiarioFiltro($compras){
        $temp = [];
        $soma = 0;
        $diaAnterior = null;
        $mesAnterior = null;
        $s = 0;

        foreach($compras as $c){
            $date = $c->created_at;
            $dia = substr($date, 8, 2);
            $mes = substr($date, 5, 2);
            if($diaAnterior != $dia){
                $temp["Dia: ".$dia."/$mes"] = $c->valor;
            }else{
                $temp["Dia: ".$diaAnterior."/$mesAnterior"] += $c->valor;
                $s += $c->valor;
            }
            $mesAnterior = $mes;
            $diaAnterior = $dia;
        }

        return $temp;
    }
    private function diferencaEntreDatas($data1, $data2){
        $dif = strtotime($data2) - strtotime($data1);
        return floor($dif / (60 * 60 * 24));
    }

    public function filtro(Request $request){
        $dataInicial = $request->data_inicial;
        $dataFinal = $request->data_final;
        $fornecedor = $request->fornecedor;
        $numero_nfe = $request->numero_nfe;
        $compras = null;
        $diferencaDatas = null;

        // if($dataInicial == null || $dataFinal == null || $fornecedor == null){
        //     session()->flash('mensagem_erro', 'Informe o fornecedor, data inicial e data final!');
        //     return redirect('/compras');
        // }

        $compras = Compra::
        select('compras.*')
        ->where('empresa_id', $this->empresa_id);

        if(($fornecedor)){
            $compras->join('fornecedors', 'fornecedors.id' , '=', 'compras.fornecedor_id')
            ->where('fornecedors.razao_social', 'LIKE', "%$fornecedor%");
        }
        if(($dataInicial) && isset($dataFinal)){
            $compras->whereBetween('compras.created_at', [
                $this->parseDate($dataInicial), 
                $this->parseDate($dataFinal, true)
            ]);
        }
        if(($numero_nfe)){
            $compras->where('nf', 'LIKE', "%$numero_nfe%");
        }

        $compras = $compras->get();

        if(isset($dataInicial) && isset($dataFinal)){
            $diferencaDatas = $this->diferencaEntreDatas($this->parseDate($dataInicial), $this->parseDate($dataFinal));
        }

        if($diferencaDatas > 31 || $diferencaDatas == null){
            $somaCompraMensal = $this->somaCompraMensalFiltro($compras);
        }else{
            $somaCompraMensal = $this->somaCompraDiarioFiltro($compras);
        }

        return view('compraManual/listAll')
        ->with('compras', $compras)
        ->with('fornecedor', $fornecedor)
        ->with('dataInicial', $dataInicial)
        ->with('numero_nfe', $numero_nfe)
        ->with('dataFinal', $dataFinal)
        ->with('somaCompraMensal', $somaCompraMensal)
        ->with('graficoJs', true)
        ->with('infoDados', "Contas filtradas")
        ->with('title', 'Filtro Compras');

    }

    private function parseDate($date, $plusDay = false){

        if($plusDay == false)
            return date('Y-m-d', strtotime(str_replace("/", "-", $date)));
        else
            return date('Y-m-d', strtotime("+1 day",strtotime(str_replace("/", "-", $date))));
    }

    public function downloadXml($id){
        $compra = Compra::
        where('id', $id)
        ->first();
        if(valida_objeto($compra)){
            $public = getenv('SERVIDOR_WEB') ? 'public/' : '';
            if($compra->nf > 0) return response()->download($public.'xml_entrada/'.$compra->chave. '.xml');
            else return response()->download($public.'xml_entrada_emitida/'.$compra->chave. '.xml');
        }else{
            return redirect('/403');
        }
    }

    public function downloadXmlCancela($id){
        $compra = Compra::
        where('id', $id)
        ->first();
        if(valida_objeto($compra)){
            $public = getenv('SERVIDOR_WEB') ? 'public/' : '';
            return response()->download($public.'xml_nfe_entrada_cancelada/'.$compra->chave. '.xml');
        }else{
            return redirect('/403');
        }
    }

    public function detalhes($id){
        $compra = Compra::
        where('id', $id)
        ->first();
        if(valida_objeto($compra)){
            $value = session('user_logged');

            return view('compraManual/detail')
            ->with('compra', $compra)
            ->with('adm', $value['adm'])
            ->with('title', 'Detalhes da compra');
        }else{
            return redirect('/403');
        }
    }

    public function delete($id){
        $compra = Compra::
        where('id', $id)
        ->first();
        if(valida_objeto($compra)){
            $stockMove = new StockMove();
            $public = getenv('SERVIDOR_WEB') ? 'public/' : '';

            if($compra->xml_path != "" && file_exists($public."xml_entrada/$compra->xml_path")){
                unlink($public."xml_entrada/$compra->xml_path");
            }
            foreach($compra->itens as $i){
        // baixa de estoque
                $stockMove->downStock($i->produto->id, $i->quantidade);
                $i->delete();
            } 

            if($compra->delete()){
                session()->flash('mensagem_sucesso', 'Registro removido!');
            }else{
                session()->flash('mensagem_erro', 'Erro!');
            }
            return redirect('/compras');
        }else{
            return redirect('/403');
        }
    }

    public function emitirEntrada($id){
        $compra = Compra::find($id);
        if(valida_objeto($compra)){
            $naturezas = NaturezaOperacao::
            where('empresa_id', $this->empresa_id)
            ->get();

            $tiposPagamento = Compra::tiposPagamento();
            return view('compraManual/emitirEntrada')
            ->with('compra', $compra)
            ->with('naturezas', $naturezas)
            ->with('tiposPagamento', $tiposPagamento)
            ->with('NFeEntradaJS', true)
            ->with('title', 'Emitir NF-e Entrada');
        }else{
            return redirect('/403');
        }
    }

    public function gerarEntrada(Request $request){
        $compra = Compra::find($request->compra_id);
        if(valida_objeto($compra)){

            $config = ConfigNota::
            where('empresa_id', $this->empresa_id)
            ->first();

            $cnpj = str_replace(".", "", $config->cnpj);
            $cnpj = str_replace("/", "", $cnpj);
            $cnpj = str_replace("-", "", $cnpj);
            $cnpj = str_replace(" ", "", $cnpj);

            $nfe_service = new NFeEntradaService([
                "atualizacao" => date('Y-m-d h:i:s'),
                "tpAmb" => (int)$config->ambiente,
                "razaosocial" => $config->razao_social,
                "siglaUF" => $config->UF,
                "cnpj" => $cnpj,
                "schemes" => "PL_009_V4",
                "versao" => "4.00",
                "tokenIBPT" => "AAAAAAA",
                "CSC" => $config->csc,
                "CSCid" => $config->csc_id
            ], 55);

            header('Content-type: text/html; charset=UTF-8');
            $natureza = NaturezaOperacao::find($request->natureza);
            
            $nfe = $nfe_service->gerarNFe($compra, $natureza, $request->tipo_pagamento);

            $signed = $nfe_service->sign($nfe['xml']);
            $resultado = $nfe_service->transmitir($signed, $nfe['chave']);
            if(substr($resultado, 0, 4) != 'Erro'){
                $compra->chave = $nfe['chave'];
            // $venda->path_xml = $nfe['chave'] . '.xml';
                $compra->estado = 'APROVADO';
                $compra->numero_emissao = $nfe['nNf'];
                $compra->data_emissao = date('Y-m-d H:i:s');
                
                $compra->save();
                $this->enviarEmailAutomatico($compra);

                $file = file_get_contents(public_path('xml_entrada_emitida/'.$nfe['chave'].'.xml'));
                importaXmlSieg($file, $this->empresa_id);

                return response()->json($resultado, 200);

            }else{
                $compra->estado = 'REJEITADO';
                $compra->save();
                return response()->json($resultado, 401);

            }
        }else{
            return response()->json("Não permitido!!", 403);

        }
    }

    public function imprimir($id){
        $compra = Compra::find($id);
        if(valida_objeto($compra)){

            $public = getenv('SERVIDOR_WEB') ? 'public/' : '';
            $xml = null;
            if(file_exists($public.'xml_entrada_emitida/'.$compra->chave.'.xml')){
                $xml = file_get_contents($public.'xml_entrada_emitida/'.$compra->chave.'.xml');
            }else if(file_exists($public.'xml_entrada/'.$compra->chave.'.xml')){
                $xml = file_get_contents($public.'xml_entrada/'.$compra->chave.'.xml');
            }else{
                session()->flash('mensagem_erro', 'Xml não encontrado!');
                return redirect('/compras');
            }
            $config = ConfigNota::
            where('empresa_id', $this->empresa_id)
            ->first();
            
            if($config->logo){
                $logo = 'data://text/plain;base64,'. base64_encode(file_get_contents($public.'logos/' . $config->logo));
            }else{
                $logo = null;
            }
        // $docxml = FilesFolders::readFile($xml);

            try {
                $danfe = new Danfe($xml);
                // $id = $danfe->monta($logo);
                $pdf = $danfe->render($logo);
                header('Content-Type: application/pdf');
                // echo $pdf;

                return response($pdf)
                ->header('Content-Type', 'application/pdf');
            } catch (InvalidArgumentException $e) {
                echo "Ocorreu um erro durante o processamento :" . $e->getMessage();
            }  
        }else{
            return redirect('/403');
        }
    }

    public function imprimirCce($id){
        $compra = Compra::
        where('id', $id)
        ->where('empresa_id', $this->empresa_id)
        ->first();

        if($compra->sequencia_cce > 0){

            $public = getenv('SERVIDOR_WEB') ? 'public/' : '';
            if(file_exists($public.'xml_nfe_entrada_correcao/'.$compra->chave.'.xml')){
                $xml = file_get_contents($public.'xml_nfe_entrada_correcao/'.$compra->chave.'.xml');

                $config = ConfigNota::
                where('empresa_id', $this->empresa_id)
                ->first();

                if($config->logo){
                    $logo = 'data://text/plain;base64,'. base64_encode(file_get_contents($public.'logos/' . $config->logo));
                }else{
                    $logo = null;
                }

                $dadosEmitente = $this->getEmitente();

                try {
                    $daevento = new Daevento($xml, $dadosEmitente);
                    $daevento->debugMode(true);
                    $pdf = $daevento->render($logo);

                    return response($pdf)
                    ->header('Content-Type', 'application/pdf');
                } catch (InvalidArgumentException $e) {
                    echo "Ocorreu um erro durante o processamento :" . $e->getMessage();
                }  
            }else{
                echo "Arquivo XML não encontrado!!";
            }
        }else{
            echo "<center><h1>Este documento não possui evento de correção!<h1></center>";
        }
    }

    private function getEmitente(){
        $config = ConfigNota::
        where('empresa_id', $this->empresa_id)
        ->first();
        return [
            'razao' => $config->razao_social,
            'logradouro' => $config->logradouro,
            'numero' => $config->numero,
            'complemento' => '',
            'bairro' => $config->bairro,
            'CEP' => $config->cep,
            'municipio' => $config->municipio,
            'UF' => $config->UF,
            'telefone' => $config->telefone,
            'email' => ''
        ];
    }

    public function cancelarEntrada(Request $request){

        $config = ConfigNota::
        where('empresa_id', $this->empresa_id)
        ->first();

        $cnpj = str_replace(".", "", $config->cnpj);
        $cnpj = str_replace("/", "", $cnpj);
        $cnpj = str_replace("-", "", $cnpj);
        $cnpj = str_replace(" ", "", $cnpj);

        $nfe_service = new NFeEntradaService([
            "atualizacao" => date('Y-m-d h:i:s'),
            "tpAmb" => (int)$config->ambiente,
            "razaosocial" => $config->razao_social,
            "siglaUF" => $config->UF,
            "cnpj" => $cnpj,
            "schemes" => "PL_009_V4",
            "versao" => "4.00",
            "tokenIBPT" => "AAAAAAA",
            "CSC" => $config->csc,
            "CSCid" => $config->csc_id
        ], 55);

        $compra = Compra::find($request->compra_id);
        $nfe = $nfe_service->cancelar($compra, $request->justificativa);
        if($this->isJson($nfe)){
            $compra->estado = 'CANCELADO';
            $compra->save();

            $file = file_get_contents(public_path('xml_nfe_entrada_cancelada/'.$compra->chave.'.xml'));
            importaXmlSieg($file, $this->empresa_id);
        }
        echo json_encode($nfe);

    }

    public function cartaCorrecao(Request $request){

        $config = ConfigNota::
        where('empresa_id', $this->empresa_id)
        ->first();

        $cnpj = str_replace(".", "", $config->cnpj);
        $cnpj = str_replace("/", "", $cnpj);
        $cnpj = str_replace("-", "", $cnpj);
        $cnpj = str_replace(" ", "", $cnpj);

        $nfe_service = new NFeEntradaService([
            "atualizacao" => date('Y-m-d h:i:s'),
            "tpAmb" => (int)$config->ambiente,
            "razaosocial" => $config->razao_social,
            "siglaUF" => $config->UF,
            "cnpj" => $cnpj,
            "schemes" => "PL_009_V4",
            "versao" => "4.00",
            "tokenIBPT" => "AAAAAAA",
            "CSC" => $config->csc,
            "CSCid" => $config->csc_id
        ], 55);

        $nfe = $nfe_service->cartaCorrecao($request->id, $request->correcao);
        echo json_encode($nfe);
    }

    private function isJson($string) {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }

    public function produtosSemValidade(){
        $produtos = Produto::select('id')
        ->where('alerta_vencimento', '>', 0)
        ->where('empresa_id', $this->empresa_id)
        ->get();

        $estoque = ItemCompra::where('validade', NULL)
        ->limit(100)->get();

        $itensSemEstoque = [];
        foreach($estoque as $e){
            foreach($produtos as $p){
                if($p->id == $e->produto_id && $e->produto->empresa_id == $this->empresa_id){
                    array_push($itensSemEstoque, $e);
                }
            }
        }
        return view('compraManual/itens_sem_estoque')
        ->with('itens', $itensSemEstoque)
        ->with('title', 'Itens sem Estoque');
    }

    public function salvarValidade(Request $request){
        $tamanhoArray = $request->tamanho_array;
        $contErro = 0;

        for($aux = 0; $aux < $tamanhoArray; $aux++){

            $validade = str_replace("/", "-", $request->input('validade_'.$aux));
            $id = $request->input('id_'.$aux);

            if(strlen($validade) == 10){ // tamanho data ok
                $item = ItemCompra::find($id);
                $dataHoje = strtotime(date('Y-m-d'));
                $validadeForm = strtotime(\Carbon\Carbon::parse($validade)->format('Y-m-d'));
                if($validadeForm > $dataHoje){ // confirma data futura
                    $item->validade = \Carbon\Carbon::parse($validade)->format('Y-m-d');
                    $item->save();
                }else{
                    $contErro++;
                }
            }else{
                $contErro++;
            }
        }
        if($contErro == 0){
            session()->flash('mensagem_sucesso', 'Validades inseridas para os itens!');
        }else{
            session()->flash('mensagem_erro', 'Erro no formulário para os itens abaixo!');
        }
        return redirect('/compras/produtosSemValidade');
        
    }

    public function validadeAlerta(){
        $dataHoje = date('Y-m-d', strtotime("-30 days",strtotime(date('Y-m-d'))));
        $dataFutura = date('Y-m-d', strtotime("+30 days",strtotime(date('Y-m-d'))));
        // $produtos = Produto::select('id')->where('alerta_vencimento', '>', 0)->get();
        $itensCompra = ItemCompra::
        whereBetween('validade', [$dataHoje, $dataFutura])
        ->limit(300)->get();
        $itens = [];
        foreach($itensCompra as $i){
            $strValidade = strtotime($i->validade);
            $strHoje = strtotime(date('Y-m-d'));
            $dif = $strValidade - $strHoje;
            $dif = $dif/24/60/60;

            if($dif <= $i->produto->alerta_vencimento && $i->produto->empresa_id == $this->empresa_id){
                array_push($itens, $i);
            }
        }

        return view('compraManual/validade_alerta')
        ->with('itens', $itens)
        ->with('title', 'Produtos com validade próxima');
    }

    public function xmlTemporaria(Request $request){
        $compra = Compra::find($request->id);

        if(valida_objeto($compra)){

            $config = ConfigNota::
            where('empresa_id', $this->empresa_id)
            ->first();

            $cnpj = str_replace(".", "", $config->cnpj);
            $cnpj = str_replace("/", "", $cnpj);
            $cnpj = str_replace("-", "", $cnpj);
            $cnpj = str_replace(" ", "", $cnpj);

            $nfe_service = new NFeEntradaService([
                "atualizacao" => date('Y-m-d h:i:s'),
                "tpAmb" => (int)$config->ambiente,
                "razaosocial" => $config->razao_social,
                "siglaUF" => $config->UF,
                "cnpj" => $cnpj,
                "schemes" => "PL_009_V4",
                "versao" => "4.00",
                "tokenIBPT" => "AAAAAAA",
                "CSC" => $config->csc,
                "CSCid" => $config->csc_id
            ], 55);

            header('Content-type: text/html; charset=UTF-8');
            $natureza = NaturezaOperacao::find($request->natureza);
            
            $nfe = $nfe_service->gerarNFe($compra, $natureza, $request->tipo_pagamento);
            if(!isset($nfe['erros_xml'])){

                $config = ConfigNota::
                where('empresa_id', $this->empresa_id)
                ->first();
                $public = getenv('SERVIDOR_WEB') ? 'public/' : '';
                
                if($config->logo){
                    $logo = 'data://text/plain;base64,'. base64_encode(file_get_contents($public.'logos/' . $config->logo));
                }else{
                    $logo = null;
                }
                
                $danfe = new Danfe($nfe['xml']);
            // $id = $danfe->monta($logo);
                $pdf = $danfe->render($logo);

            // return response($pdf)
            // ->header('Content-Type', 'application/pdf');

                return response($nfe['xml'])
                ->header('Content-Type', 'application/xml');
            }else{
                print_r($nfe['erros_xml']);
            }
        }
    }

    public function danfeTemporaria(Request $request){
        $compra = Compra::find($request->id);

        if(valida_objeto($compra)){

            $config = ConfigNota::
            where('empresa_id', $this->empresa_id)
            ->first();

            $cnpj = str_replace(".", "", $config->cnpj);
            $cnpj = str_replace("/", "", $cnpj);
            $cnpj = str_replace("-", "", $cnpj);
            $cnpj = str_replace(" ", "", $cnpj);

            $nfe_service = new NFeEntradaService([
                "atualizacao" => date('Y-m-d h:i:s'),
                "tpAmb" => (int)$config->ambiente,
                "razaosocial" => $config->razao_social,
                "siglaUF" => $config->UF,
                "cnpj" => $cnpj,
                "schemes" => "PL_009_V4",
                "versao" => "4.00",
                "tokenIBPT" => "AAAAAAA",
                "CSC" => $config->csc,
                "CSCid" => $config->csc_id
            ], 55);

            header('Content-type: text/html; charset=UTF-8');
            $natureza = NaturezaOperacao::find($request->natureza);
            
            $nfe = $nfe_service->gerarNFe($compra, $natureza, $request->tipo_pagamento);
            if(!isset($nfe['erros_xml'])){

                $config = ConfigNota::
                where('empresa_id', $this->empresa_id)
                ->first();
                $public = getenv('SERVIDOR_WEB') ? 'public/' : '';

                if($config->logo){
                    $logo = 'data://text/plain;base64,'. base64_encode(file_get_contents($public.'logos/' . $config->logo));
                }else{
                    $logo = null;
                }

                $danfe = new Danfe($nfe['xml']);
            // $id = $danfe->monta($logo);
                $pdf = $danfe->render($logo);

                return response($pdf)
                ->header('Content-Type', 'application/pdf');
            }else{
                print_r($nfe['erros_xml']);
            }

            // return response($nfe['xml'])
            // ->header('Content-Type', 'application/xml');
        }
    }

    public function consultar(Request $request){

        $config = ConfigNota::
        where('empresa_id', $this->empresa_id)
        ->first();

        $compra = Compra::find($request->compra_id);

        $cnpj = str_replace(".", "", $config->cnpj);
        $cnpj = str_replace("/", "", $cnpj);
        $cnpj = str_replace("-", "", $cnpj);
        $cnpj = str_replace(" ", "", $cnpj);

        $nfe_service = new NFeEntradaService([
            "atualizacao" => date('Y-m-d h:i:s'),
            "tpAmb" => (int)$config->ambiente,
            "razaosocial" => $config->razao_social,
            "siglaUF" => $config->UF,
            "cnpj" => $cnpj,
            "schemes" => "PL_009_V4",
            "versao" => "4.00",
            "tokenIBPT" => "AAAAAAA",
            "CSC" => $config->csc,
            "CSCid" => $config->csc_id
        ], 55);

        $c = $nfe_service->consultar($compra);
        echo json_encode($c);

    }

    public function salvarChaveRef(Request $request){
        try{

            CompraReferencia::create([
                'compra_id' => $request->compra_id,
                'chave' => $request->chave
            ]);
            session()->flash('mensagem_sucesso', "Chave referenciada!");
        }catch(\Exception $e){
            session()->flash('mensagem_erro', "Erro: " . $e->getMessage());
        }
        return redirect()->back();
    }

    public function deleteChave($id){
        try{

            CompraReferencia::find($id)->delete();
            session()->flash('mensagem_sucesso', "Chave removida!");
        }catch(\Exception $e){
            session()->flash('mensagem_erro', "Erro: " . $e->getMessage());
        }
        return redirect()->back();
    }

    private function enviarEmailAutomatico($compra){
        $escritorio = EscritorioContabil::
        where('empresa_id', $this->empresa_id)
        ->first();

        if($escritorio != null && $escritorio->envio_automatico_xml_contador){
            $email = $escritorio->email;
            Mail::send('mail.xml_automatico', ['descricao' => 'Envio de NF-e Entrada'], function($m) use ($email, $compra){
                $nomeEmpresa = getenv('MAIL_NAME');
                $nomeEmpresa = str_replace("_", " ",  $nomeEmpresa);
                $nomeEmpresa = str_replace("_", " ",  $nomeEmpresa);
                $emailEnvio = getenv('MAIL_USERNAME');

                $m->from($emailEnvio, $nomeEmpresa);
                $m->subject('Envio de XML Automático');

                $m->attach(public_path('xml_entrada_emitida/'.$compra->chave.'.xml'));
                $m->to($email);
            });
        }
    }

    public function estadoFiscal($id){
        $compra = Compra::
        where('id', $id)
        ->first();
        $value = session('user_logged');
        if($value['adm'] == 0) return redirect()->back();
        if(valida_objeto($compra)){

            return view("compraManual/alterar_estado_fiscal")
            ->with('compra', $compra)
            ->with('title', "Alterar estado compra $id");
        }else{
            return redirect('/403');
        }
    }

    public function estadoFiscalStore(Request $request){
        try{
            $compra = Compra::find($request->compra_id);
            $estado = $request->estado;
            
            $compra->estado = $estado;
            if ($request->hasFile('file')){
                $public = getenv('SERVIDOR_WEB') ? 'public/' : '';

                $xml = simplexml_load_file($request->file);
                $chave = substr($xml->NFe->infNFe->attributes()->Id, 3, 44);
                $file = $request->file;
                $file->move(public_path('xml_entrada_emitida'), $chave.'.xml');
                $compra->chave = $chave;
                $compra->numero_emissao = (int)$xml->NFe->infNFe->ide->nNF;
                
            }

            $compra->save();
            session()->flash("mensagem_sucesso", "Estado alterado");

        }catch(\Exception $e){
            session()->flash("mensagem_erro", "Erro: " . $e->getMessage());

        }
        return redirect()->back();
    }

    public function setNaturezaPagamento(Request $request){
        $compra = Compra::find($request->id);

        $compra->tipo_pagamento = $request->tipo_pagamento;
        $compra->natureza_id = $request->natureza_id;
        $compra->save();
        return "sucesso";
    }

    public function print($id){
        $compra = Compra::find($id);
        if(valida_objeto($compra)){
            $config = ConfigNota::
            where('empresa_id', $this->empresa_id)
            ->first();
            $p = view('compraManual/print')
            ->with('config', $config)
            ->with('compra', $compra);
            // return $p;

            $domPdf = new Dompdf(["enable_remote" => true]);
            $domPdf->loadHtml($p);

            $pdf = ob_get_clean();

            $domPdf->setPaper("A4");
            $domPdf->render();
            $domPdf->stream("Pedido de Compra $id.pdf");
        }else{
            return redirect('/403');
        }
    }
}
